﻿using System;
using System.Net;
using System.Threading.Tasks;
using DotNetCore.CAP.Dashboard;
using Microsoft.AspNetCore.Http;

namespace DotNetCore.CAP
{
    public class DashboardMiddleware
    {
        private readonly DashboardOptions _options;
        private readonly RequestDelegate _next;
        private readonly IStorage _storage;
        private readonly RouteCollection _routes;

        public DashboardMiddleware(RequestDelegate next, DashboardOptions options, IStorage storage, RouteCollection routes)
        {
            _next = next ?? throw new ArgumentNullException(nameof(next));
            _options = options ?? throw new ArgumentNullException(nameof(options));
            _storage = storage ?? throw new ArgumentNullException(nameof(storage));
            _routes = routes ?? throw new ArgumentNullException(nameof(routes));
        }

        public Task Invoke(HttpContext context)
        {
            PathString matchedPath;
            PathString remainingPath;

            if (context.Request.Path.StartsWithSegments(_options.PathMatch, out matchedPath, out remainingPath))
            {
                // Update the path
                var path = context.Request.Path;
                var pathBase = context.Request.PathBase;
                context.Request.PathBase = pathBase.Add(matchedPath);
                context.Request.Path = remainingPath;

                try
                {
                    var dashboardContext = new CapDashboardContext(_storage, _options, context);
                    var findResult = _routes.FindDispatcher(context.Request.Path.Value);

                    if (findResult == null)
                    {
                        return _next.Invoke(context);
                    }

                    foreach (var filter in _options.Authorization)
                    {
                        if (!filter.Authorize(dashboardContext))
                        {
                            var isAuthenticated = context.User?.Identity?.IsAuthenticated;

                            context.Response.StatusCode = isAuthenticated == true
                                ? (int)HttpStatusCode.Forbidden
                                : (int)HttpStatusCode.Unauthorized;

                            return Task.CompletedTask;
                        }
                    }

                    dashboardContext.UriMatch = findResult.Item2;

                    return findResult.Item1.Dispatch(dashboardContext);
                }
                finally
                {
                    context.Request.PathBase = pathBase;
                    context.Request.Path = path;
                }
            }
            else
            {
                return _next(context);
            }
        }
    }
}
