﻿using MongoDB.Driver;
using MongoDB.Driver.Linq;
using Plus.Dependency;
using Plus.Domain.Entities;
using Plus.Domain.Repositories;
using System.Linq;

namespace Plus.MongoDb.Repositories
{
    /// <summary>
    /// MongoDB Repository
    /// </summary>
    /// <typeparam name="TEntity"></typeparam>
    public abstract class MongoDbRepositoryBase<TEntity> : MongoDbRepositoryBase<TEntity, int>, IRepository<TEntity>, IRepository<TEntity, int>, IRepository, ITransientDependency where TEntity : class, IEntity<int>
    {
        public MongoDbRepositoryBase(IMongoDatabaseProvider databaseProvider)
            : base(databaseProvider)
        {
        }
    }

    /// <summary>
    /// MongoDB Repository
    /// </summary>
    /// <typeparam name="TEntity"></typeparam>
    /// <typeparam name="TPrimaryKey"></typeparam>
    public abstract class MongoDbRepositoryBase<TEntity, TPrimaryKey> : PlusRepositoryBase<TEntity, TPrimaryKey> where TEntity : class, IEntity<TPrimaryKey>
    {
        private readonly IMongoDatabaseProvider _databaseProvider;

        public virtual IMongoDatabase Database => _databaseProvider.Database;

        public virtual IMongoCollection<TEntity> Collection
        {
            get
            {
                string name = CollectionName;
                if (CollectionName.IsNullOrEmpty())
                {
                    name = typeof(TEntity).Name;
                }
                return _databaseProvider.Database.GetCollection<TEntity>(name, null);
            }
        }

        public abstract string CollectionName { get; }

        public MongoDbRepositoryBase(IMongoDatabaseProvider databaseProvider)
        {
            _databaseProvider = databaseProvider;
        }

        public override IQueryable<TEntity> GetAll()
        {
            return Collection.AsQueryable();
        }

        public override TEntity Get(TPrimaryKey id)
        {
            var query = MongoDB.Driver.Builders.Query<TEntity>.EQ(e => e.Id, id);
            var entity = Collection.Find(query);
            if (entity.IsNull())
            {
                throw new EntityNotFoundException("There is no such an entity with given primary key. Entity type: " + typeof(TEntity).FullName + ", primary key: " + id);
            }

            return entity;
        }

        public override TEntity FirstOrDefault(TPrimaryKey id)
        {
            var query = MongoDB.Driver.Builders.Query<TEntity>.EQ(e => e.Id, id);
            return Collection.FindOne(query);
        }

        public override TEntity Insert(TEntity entity)
        {
            Collection.Insert(entity);
            return entity;
        }

        public override TEntity Update(TEntity entity)
        {
            Collection.Save(entity);
            return entity;
        }

        public override void Delete(TEntity entity)
        {
            Delete(entity.Id);
        }

        public override void Delete(TPrimaryKey id)
        {
            var query = MongoDB.Driver.Builders.Query<TEntity>.EQ(e => e.Id, id);
            Collection.Remove(query);
        }
    }
}